// 内容脚本
class ContentScript {
  constructor() {
    this.init();
  }

  init() {
    // 监听来自 popup 的消息
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      console.log('收到消息:', request);
      
      if (request.type === 'GET_SELECTED_TEXT') {
        const selectedText = this.getSelectedText();
        console.log('选中的文本:', selectedText);
        
        sendResponse({ text: selectedText });
        return true;
      }
    });
    
    // 监听选中文本变化
    document.addEventListener('selectionchange', () => {
      const selectedText = this.getSelectedText();
      console.log('文本选择变化:', selectedText);
      
      if (selectedText) {
        this.saveSelectedText(selectedText).catch(error => {
          // 如果是扩展上下文失效，则重新初始化
          if (error.message.includes('Extension context invalidated')) {
            console.log('扩展上下文失效，尝试重新初始化...');
            this.init();
          }
        });
      }
    });
  }

  getSelectedText() {
    return window.getSelection().toString().trim();
  }

  async saveSelectedText(text) {
    if (!text) return;
    
    try {
      // 检查扩展上下文是否有效
      if (!chrome.runtime?.id) {
        throw new Error('Extension context invalidated');
      }

      await chrome.storage.local.set({
        selectedText: text,
        timestamp: Date.now()
      });
      
      // 通知 popup 更新
      chrome.runtime.sendMessage({
        type: 'TEXT_SELECTED',
        data: { text }
      });
      console.log('已保存文本:', text);
    } catch (error) {
      console.error('保存选中文本失败:', error);
      throw error; // 重新抛出错误以便外层处理
    }
  }
}

// 初始化内容脚本
console.log('内容脚本已初始化');
new ContentScript(); 